<?php if (!defined('BASEPATH')) exit('No direct script access allowed');
/**
 * BackendPro
 *
 * A website backend system for developers for PHP 4.3.2 or newer
 *
 * @package			BackendPro
 * @author			Adam Price
 * @copyright		Copyright (c) 2008
 * @license			http://www.gnu.org/licenses/lgpl.html
 * @tutorial		BackendPro.pkg
 */

/**
 * MODULE NAME   : MY_Validation.php
 *
 * DESCRIPTION   : MY validation library - items added for ConCentric
 *
 * MODIFICATION HISTORY
 *   V1.0   2008-11-11 13:00 PM   - Leane Verhulst     - Created
 *
 * @package			ConCentric
 * @subpackage		my form helper Class
 * @author			Leane Verhulst
 * @copyright		Copyright (c) 2008
 * @license			http://www.gnu.org/licenses/gpl.html
 */

 // ---------------------------------------------------------------------------

/**
 * MY_Validation
 *
 * Implements custom validation functions for use
 * when validation forms in BackendPro
 *
 * @package			BackendPro
 * @subpackage		Libraries
 * @tutorial		MY_Validation.cls
 */
class MY_Validation extends CI_Validation
{
	/**
	 * Constructor
	 */
	function MY_Validation()
	{
		// Extend parent
		parent::CI_Validation();

		// Get CI Instance
		$this->CI = &get_instance();

		// Load the language file containing custom error messages
		$this->CI->lang->load('error');

		log_message('debug','MY_Validation Class Initialized');
	}

	/**
	 * Set Default Value
	 * 
	 * Assigns a default value to a form field
	 * 
	 * @access public
	 * @param mixed $data Field name OR Array
	 * @param mixed $value Field value
	 * @return void 
	 */
	function set_default_value($data=NULL, $value=NULL)
	{
		if (is_array($data))
		{
			foreach($data as $field => $value)
			{
				$this->set_default_value($field,$value);
			}
			return;
		}

		$this->$data    = $value;
		$_POST[$data]   = $value;
		return;
	}


	/**
	 * Output Validation Errors
	 *
	 * Using the Status class move all errors into an error
	 * message
	 *
	 * @access public
	 * @return void
	 */
	function output_errors()
	{
		// Make sure the status module is
		$this->CI->load->module_library('status','status');

		foreach ( $this->_error_array as $error )
		{
			flashMsg('warning',$error);
		}
		return;
	}

	/**
	 * Check for valid captcha
	 * 
	 * Contact the ReCaptcha server and check the input is valid
	 * 
	 * @access public
	 * @return boolean
	 */
	function valid_captcha()
	{
		// Makre sure auth library is loaded
		$this->CI->load->module_library('auth','Userlib');

		// Make sure the captcha library is loaded
		$this->CI->load->module_library('recaptcha','Recaptcha');

		// Set the error message
		$this->CI->validation->set_message('valid_captcha', $this->CI->lang->line('userlib_validation_captcha'));

		// Perform check
		$this->CI->recaptcha->recaptcha_check_answer($this->CI->input->server('REMOTE_ADDR'), $this->CI->input->post('recaptcha_challenge_field'), $this->CI->input->post('recaptcha_response_field'));

		return $this->CI->recaptcha->is_valid;
	}

	/**
	 * Check that the username is spare
	 * 
	 * Check that the username given is not in use
	 * 
	 * @access public
	 * @param string $username Username
	 * @return boolean
	 */
	function spare_username($username)
	{
		// Makre sure auth library is loaded
		$this->CI->load->module_library('auth','Userlib');

		$query = $this->CI->user_model->fetch('Users',NULL,NULL,array('username'=>$username)); 

		// Set the error message
		$this->CI->validation->set_message('spare_username', $this->CI->lang->line('userlib_validation_username'));

		return ($query->num_rows() == 0) ? TRUE : FALSE; 
	}

	/**
	 * Check that the email is spare
	 * 
	 * Check that the username given is not in use by another user
	 * 
	 * @access public
	 * @param string $email Email
	 * @retrun boolean
	 */
	function spare_email($email)
	{
		// Makre sure auth library is loaded
		$this->CI->load->module_library('auth','Userlib');  

		$query = $this->CI->user_model->fetch('Users',NULL,NULL,array('email'=>$email));

		// Set the error message
		$this->CI->validation->set_message('spare_email', $this->CI->lang->line('userlib_validation_email'));

		return ($query->num_rows() == 0) ? TRUE : FALSE;
	}

	/**
	 * Check Spare Username
	 * 
	 * When modifying a user check the username is spare
	 * 
	 * @access public
	 * @param string $username Username
	 * @return boolean
	 */
	function spare_edit_username($username)
	{
		// Makre sure auth library is loaded
		$this->CI->load->module_library('auth','Userlib');

		$query = $this->CI->user_model->fetch('Users',NULL,NULL,array('username'=>$username,'id!='=>$this->CI->input->post('id'))); 

		// Set the error message
		$this->CI->validation->set_message('spare_edit_username', $this->CI->lang->line('userlib_validation_username'));

		return ($query->num_rows() == 0) ? TRUE : FALSE; 
	}

	/**
	 * Check Spare Email
	 * 
	 * When modifying a user check the email is spare
	 * 
	 * @access public
	 * @param string $email Email
	 * @retrun boolean
	 */
	function spare_edit_email($email)
	{
		// Makre sure auth library is loaded
		$this->CI->load->module_library('auth','Userlib');  

		$query = $this->CI->user_model->fetch('Users',NULL,NULL,array('email'=>$email,'id!='=>$this->CI->input->post('id')));

		// Set the error message
		$this->CI->validation->set_message('spare_edit_email', $this->CI->lang->line('userlib_validation_email'));

		return ($query->num_rows() == 0) ? TRUE : FALSE;
	}

	/**
	 * numeric rules
	 */
	function greater_than($str,$min)
	{
		if(!is_numeric($str)){ return FALSE; }
		return $str > $min;
	}

	function less_than($str,$max)
	{
		if(!is_numeric($str)){ return FALSE; }
		return $str < $max;
	}

	function equal_to($str,$eq)
	{
		if(!is_numeric($str)){ return FALSE; }
		return $str == $eq;
	}

	/**
	 *  checkbox rules
	 *
	 * use with required rule
	 */
	function max_count($str,$max)
	{
		return (count($str) <= $max);
	}

	function min_count($str,$min)
	{
		return count($str) >= $min;
	}

	/**
	 * validate date from text input
	 *   Parameters control date delimiter and part order.
	 *   A date of all zeros is allowed.
	 *
	 * example : $rule['date'] = 'valid_textdate[-,2,1,0]';
	 */
	function valid_textdate($str,$params = '')
	{
		if($params == '')
		{
			// default setting
			$divider = '-';
			$yearpart = 0;
			$monthpart = 1;
			$daypart = 2;
		}
		else
		{
			$explode = explode(',',$params);
			$divider = $explode[0];
			$yearpart = $explode[1];
			$monthpart = $explode[2];
			$daypart = $explode[3];
		}
		$explode2 = explode($divider,$str);
		if(count($explode2) != 3)
		{
			return FALSE;
		}
		$year = $explode2[$yearpart];
		$month = $explode2[$monthpart];
		$day = $explode2[$daypart];
		if (is_numeric($year) && is_numeric($month) && is_numeric($day))
		{
			if ($year == '0000' && $month == '00' && $day == '00')
			{
				return TRUE;
			}
			else
			{
				return checkdate($month,$day,$year);
			}
		}
		return FALSE;
	}

	/**
	 * validate time from text input
	 *   A time of all zeros is allowed.
	 *
	 * example : $rule['time'] = 'valid_texttime';
	 */
	function valid_texttime($str)
	{
		$divider = ':';
		$hourpart = 0;
		$minutepart = 1;
		$secondpart = 2;
		$explode2 = explode($divider,$str);
		if(count($explode2) != 3)
		{
			return FALSE;
		}
		$hour = $explode2[$hourpart];
		$minute = $explode2[$minutepart];
		$second = $explode2[$secondpart];
		if (is_numeric($hour) && is_numeric($minute) && is_numeric($second))
		{
			if ($hour > -1 && $hour < 24 && $minute > -1 && $minute < 60 && $second > -1 && $second < 60)
			{
				return TRUE;
			}
		}
		return FALSE;
	}

	/**
	 * Validate date and time from text input
	 *   Parameters control date delimiter and part order.
	 *   A date/time of all zeros is allowed.
	 *
	 * example : $rule['date'] = 'valid_textdatetime[-,2,1,0]';
	 */
	function valid_textdatetime($str,$params = '')
	{
		//split the string into 2 parts - date and time
		$dttm = explode(" ", $str);

		if ($this->valid_textdate($dttm[0], $params))
		{
			if ($this->valid_texttime($dttm[1]))
			{
				return TRUE;
			}
		}
		return FALSE;
	}

	/**
	 * Alpha-numeric with underscores, dashes and spaces
	 *
	 * @access    public
	 * @param     string
	 * @return    bool
	 */
	function alpha_dash_space($str)
	{
		return ( ! preg_match("/^([-a-z0-9_ -])+$/i", $str)) ? FALSE : TRUE;
	}

	/**
	 * Check for a valid person name. Can contain alpha-numeric and certain punctuation.
	 *
	 * @access    public
	 * @param     string
	 * @return    bool
	 */
	function valid_person_name($str)
	{
		return ( ! preg_match("/^([-a-z0-9_ ,.'()-])+$/i", $str)) ? FALSE : TRUE;
	}

	/**
	 * Color Hex
	 *
	 * #ffffff
	 *
	 * @access    public
	 * @param    string
	 * @return    bool
	 */
	function color_hex($hex)
	{
		return ( preg_match("/#[a-fA-F0-9]{6}/", $hex)) ? TRUE : FALSE;
	}


}


/* End of file MY_Validation.php */
/* Location: ./system/application/libraries/MY_Validation.php */